<!---
CMake build script for ACM Collected Algorithms: DCUHRE and DQUAD
Copyright (C) 2024  Tim Kaune

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as
published by the Free Software Foundation, either version 3 of the
License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this program.  If not, see
<https://www.gnu.org/licenses/>.
--->

# ACM Collected Algorithms: DCUHRE and DQUAD #

This project provides a CMake framework, that automatically downloads the two
Fortran integrator libraries DCUHRE and DQUAD from the ACM Collected Algorithms
(<https://calgo.acm.org/>), converts them to C and builds them.

## Prerequisites ##

What you need:

- [Git](https://git-scm.com/) (optional) for getting the code and contributing
  to the project
- [CMake](https://cmake.org/) and [Ninja](https://ninja-build.org/) for building
  the project
- C compiler
  - GCC for Linux (<https://gcc.gnu.org/>)
  - Clang for Mac OS X (<https://clang.llvm.org/>)
  - Cygwin + GCC for Windows (<https://www.cygwin.com/>)
- NetlibF2C (<https://gitlab.aei.uni-hannover.de/netlib/netlib-f2c>)
- LoadStaticSharedTargets (<https://gitlab.aei.uni-hannover.de/cmake/loadstaticsharedtargets>)

### Linux (Ubuntu) ###

Use your distribution's package manager to install the necessary packages:

```shell
$ sudo apt-get install cmake ninja-build gcc
```

### Mac OS X ###

You will need one of the community package managers for Mac OS X:
[Homebrew](https://brew.sh/) or
[MacPorts](https://www.macports.org/install.php). For installing one of these,
please refer to their respective installation instructions.

#### Homebrew ####

```shell
$ brew install cmake ninja llvm
```

#### MacPorts ####

```shell
$ sudo port -vb install cmake-devel ninja clang-<XX>
```

where you have to replace `<XX>` with the latest stable version of Clang.
You can discover all available versions of Clang like this:

```shell
$ port search --name --line --regex '^clang-?\d*\.?\d*$'
```

### Windows ###

The easiest thing to do is using the [Windows Subsystem for Linux
(WSL)](https://learn.microsoft.com/en-us/windows/wsl/install) and follow the
Linux instructions above.

Otherwise, install [Git for Windows](https://gitforwindows.org/) for version
control and [cygwin](https://cygwin.com/install.html), which is a large
collection of GNU and Open Source tools which provide functionality similar to a
Linux distribution on Windows. During the `cygwin` installation you'll be
prompted to add additional packages. Search and select the following:

- `cmake`, `ninja` and `gcc-core` for the compiler and the build system

After `cygwin` finishes installing, use the cygwin terminal to start the build
process.

## How to compile ##

1. (Optional) First, build the LoadStaticSharedTargets and NetlibF2C projects
   and install them either system-wide or in a central location, e. g. somewhere
   in your home directory. Please refer to the project READMEs for build
   instructions. If you skip this step, LoadStaticSharedTargets and NetlibF2C
   should automatically be downloaded and included in your build. But you might
   end up compiling some parts multiple times unnecessarily.
2. Clone this git repository or download the source code archive and unpack it
   to an arbitrary directory (e.g. `calgo-integrators`).
3. Go to this directory and type `cmake --list-presets`. A list of available
   build configurations will be shown to you.
4. For preparing a single configuration with `ninja`, type `cmake --preset
   ninja-shared-release`, if you installed LoadStaticSharedTargets and NetlibF2C
   system-wide or not at all. Otherwise, type

   ```shell
   $ cmake --preset ninja-shared-release -D "CMAKE_PREFIX_PATH=/path/to/NetlibF2C/install/prefix;/path/to/LoadStaticSharedTargets/install/prefix"
   ```

   Replace the paths to the LoadStaticSharedTargets and NetlibF2C install
   prefixes with the actual paths on your system! The license of the ACM
   Collected Algorithms (CALGO) will be shown to you. If you accept this
   license, add the required variable to the previous command:

   ```shell
   $ cmake --preset ninja-shared-release -D CalgoIntegrators_LICENSE_ACCEPTED=YES
   ```

   If not installed previously, you will also have to accept NetlibF2C's license
   in a similar way. This will populate the build directory with a CMake
   configuration tree. You can also prepare all shared or static configurations
   at once by using `cmake --preset ninja-multi-shared`. By default, the
   configured installation directory is `./install`. You can specify a different
   install location by setting the `CMAKE_INSTALL_PREFIX` variable:

   ```shell
   $ cmake --preset ninja-shared-release -D "CMAKE_INSTALL_PREFIX=/path/to/install/prefix"
   ```

   The `-D` option can be given multiple times, so you can combine
   `CMAKE_PREFIX_PATH`, `CalgoIntegrators_LICENSE_ACCEPTED` and
   `CMAKE_INSTALL_PREFIX`.
5. Now, you can build with `cmake --build --preset ninja-shared-release`. You
   should see some informative terminal output. If you used a Ninja Multi-Config
   in the previous step, you have to determine the config to build now: `cmake
   --build --preset ninja-multi-shared-release` (repeat with the debug config).
   See all available build presets with `cmake --build --list-presets`.
6. (Optional) Build and run the tests with `cmake --build --preset
   ninja-shared-release --target check` (or for Ninja Multi-Config `cmake
   --build --preset ninja-multi-shared-release` --target check).
7. Finally, install the built artifacts to the configured install prefix with
   `cmake --build --preset ninja-shared-release --target install`. If the
   configured install prefix is a system-wide location (like `/usr/local`),
   installing might require `sudo`. If you used a Ninja Multi-Config in step 4,
   use the respective build preset for installing, too: `cmake --build --preset
   ninja-multi-shared-release --target install` (repeat with the debug config).
   You can also install either only the DCUHRE library or only the DQUAD library
   with `cmake --build --preset ninja-shared-release --target install
   --component dcuhre` or `cmake --build --preset ninja-shared-release --target
   install --component dquad`
8. Repeat this process for the single config presets `ninja-static-release`,
   `ninja-shared-debug` and `ninja-static-debug` or for the other multi-config
   preset.

Now, the integrator code is compiled and you can use it in other projects. The
installed integrator package is discoverable by CMake as __CalgoIntegrators__.
In the `CMakeLists.txt` of the other project, call:

```cmake
include(FetchContent)

FetchContent_Declare(
    CalgoIntegrators
    GIT_REPOSITORY "git@gitlab.aei.uni-hannover.de:netlib/calgo-integrators.git"
    GIT_TAG v1.7.0
    SYSTEM
    FIND_PACKAGE_ARGS 1.7.0 CONFIG NAMES CalgoIntegrators
)
set(CalgoIntegrators_INCLUDE_PACKAGING TRUE)
FetchContent_MakeAvailable(CalgoIntegrators)
```

This discovers an installed version of the CalgoIntegrators package or adds it
to the other project's install targets. To help CMake discover the
CalgoIntegrators package, call CMake for the other project like this:

```shell
$ cmake -B ./build -D "CMAKE_PREFIX_PATH=/path/to/CalgoIntegrators/install/prefix"
```

Replace the path to the CalgoIntegrators install prefix with the actual path on
your system! If CalgoIntegrators is installed in a proper system-wide location,
the `CMAKE_PREFIX_PATH` shouldn't be necessary.

This makes two CMake targets available to you, the `Netlib::DCUHRE` and the
`Netlib::DQUAD` library targets (static or shared depends on the value of
`BUILD_SHARED_LIBS` in the other project). The `Netlib::libf2c` is already
contained in these targets.

For more information read the two papers referenced below.

## References ##

Paola Favati, Grazia Lotti, and Francesco Romani. 1991. Algorithm 691: Improving
QUADPACK automatic integration routines. ACM Trans. Math. Softw. 17, 2 (June
1991), 218–232. DOI:<https://dl.acm.org/doi/10.1145/108556.108580>

Jarle Berntsen, Terje O. Espelid, and Alan Genz. 1991. Algorithm 698: DCUHRE: an
adaptive multidemensional integration routine for a vector of integrals. ACM
Trans. Math. Softw. 17, 4 (Dec. 1991), 452–456.
DOI:<https://dl.acm.org/doi/10.1145/210232.210234>
